<?php // $Id: class.socket.php,v 1.2 2007/05/05 11:18:00 janne Exp $

/**
 * Connection_socket sub class
 *
 * This file contains an interface for socket functions
 * used in connection class.
 *
 * @package Connection
 * @author Janne Mikkonen
 * @version 1.1
 * @copyright 2007 - 2011 Janne Mikkonen <janne dot mikkonen at julmajanne dot com>
 * @license http://www.opensource.org/licenses/mit-license.php MIT License
 */

/**
 * Connection_socket sub class
 *
 * @package Connection
 * @author Janne Mikkonen
 * @author $Author: janne $
 * @version $Revision: 1.2 $
 * @copyright 2007 - 2011 Janne Mikkonen <janne dot mikkonen at julmajanne dot com>
 * @license http://www.opensource.org/licenses/mit-license.php MIT License
 */
class Connection_socket extends Connection {

    private $domain   = AF_INET;
    private $type     = SOCK_STREAM;
    private $protocol = SOL_TCP;
    private $sock     = null;

    /**
     * PHP 4 constructor.
     *
     * @param string $host Hostname or IP-address
     * @param int $port Port number
     * @param string $type interface name
     */
    public function __construct($host, $port, $type) {

        if ( !function_exists('socket_create') ) {
            trigger_error('Socket extension is not available!');
            return false;
        }

        parent::Setup($type);
        $this->_host = $host;
        $this->_port = $port;

    }

    /**
     * Set address/protocol family.
     *
     * Available address/protocol families:
     * AF_INET, AF_INET6 and AF_UNIX
     *
     * @param int $domain
     * @return void
     */
    public function setDomain($domain) {
        if ( $domain & AF_UNIX ) {
            $this->domain = AF_UNIX;
        } else if ( $domain & AF_INET6 ) {
            $this->domain = AF_INET6;
        } else {
            $this->domain = AF_INET;
        }
    }

    /**
     * Set socket type
     *
     * @param int $type
     * @return void
     */
    public function setType ($type) {
        switch ( intval($type) ) {
            case SOCK_DGRAM: $this->type = SOCK_DGRAM; break;
            case SOCK_SEQPACKET: $this->type = SOCK_SEQPACKET; break;
            case SOCK_RAW: $this->type = SOCK_RAW; break;
            case SOCK_RDM: $this->type = SOCK_RDM; break;
        }
    }

    /**
     * Create a socket
     *
     * @access private
     * @return mixed Returns a resource on success an false when failed
     */
    private function _create () {
        $socket = @socket_create($this->domain, $this->type, $this->protocol);
        if ( is_resource($socket) ) {
            return $socket;
        } else {
            throw new Exception("Cannot create connection socket!");
        }
        return false;
    }

    /**
     * Bind connection to address
     *
     * @access private
     * @param string $address
     * @param int $port
     * @return bool
     */
    private function _bind ($address, $port=null) {
        if ( $this->sock ) {
            if ( !is_null($port) ) {
                return @socket_bind($this->sock, $address, $port);
            }
            return @socket_bind($this->sock, $address);
        } else {
            throw new Exception("Cannot bind to {$address}!");
        }
        return false;
    }

    /**
     * Listen requested resource.
     *
     * @access private
     * @param int $backlog
     */
    private function _listen($backlog=5) {
        if ( $this->sock &&
           ($this->type === SOCK_STREAM or $this->type === SOCK_SEQPACKET) ) {
            return @socket_listen($this->sock, $backlog);
        }
    }

    /**
     * Open connection
     *
     * Opens a connection to required host
     * @return mixed
     */
    public function Open () {
        $this->sock = $this->_create();
        if ( $this->sock < 0 ) {
            $this->_errnro = socket_last_error();
            $this->_errstr = socket_strerror($this->_errnro);
            throw new Exception("Socket create failed: {$this->_errstr}");
        }

        if ( !@socket_connect($this->sock, $this->_host, $this->_port) ) {
            $this->_errnro = socket_last_error();
            $this->_errstr = socket_strerror($this->_errnro);

            throw new Exception("Could not connect to {$this->_host}: {$this->_errstr}");
        }
        return true;
    }

    /**
     * Close connection.
     * @return void
     */
    public function Close () {
        if ( is_resource($this->sock) ) {
            socket_close($this->sock);
            return true;
        }
        return false;
    }

    /**
     * Write data
     *
     * Send data to server.
     * @param string $data Data to send
     * @return bool
     */
    public function Write ($data) {
        if ( is_resource($this->sock) ) {
            return socket_write($this->sock, $data, strlen($data));
        } else {
            throw new Exception("Connection closed! Cannot write to socket!");
        }
        return false;
    }

    /**
     * Read data from connection
     *
     * @param $length
     * @return string
     */
    public function Read ($length=512, $type=PHP_BINARY_READ) {
        if ( is_resource($this->sock) ) {
            $out = '';
            while ( $buf = socket_read($this->sock, $length, $type) ) {
                $out .= $buf;
            }
            return $out;
        } else {
            throw new Exception("Connection closed! Cannot read from socket!");
        }
    }

    /**
     * Override parent method
     *
     * @see Connection::setSecure()
     */
    public function setSecure ($type='ssl') {
        return;
    }

}
?>
